/**
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * IN NO EVENT SHALL THE SOFTCORPORATION BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION).
 *
 */
package com.softcorporation.nlp.wordnet;

import java.io.*;

import org.json.JSONArray;
import org.json.JSONObject;

/**
 * WordNet JSON Demo
 * @author Vadim Permakoff
 * @version: $Revision:   1.0  $
 */
public class WordNetDemo
{
    /**
     * Main method
     */
    public static void main(String[] args)
    {
//        args = new String[] {"/tmp/wn.json"};

        if (args.length == 0)
        {
            log("Specify the wordnet json file");
            return;
        }
        String inputFileName = args[0];

        try
        {
            File inputFile = new File(inputFileName);
            String jsonString = readFile(inputFile, "UTF-8");
            JSONObject wn = new JSONObject(jsonString);
            log("Loaded dataset: " + wn.getString("dataset"));

            JSONObject words = wn.getJSONObject("words");
            JSONObject synsets = wn.getJSONObject("synsets");

            BufferedReader keyboardInput = new BufferedReader(new InputStreamReader(System.in));
            while (true)
            {
                System.out.print("\nPlease enter a text ('Enter' - exit): ");
                String word = keyboardInput.readLine();

                if (word.trim().length() == 0)
                {
                    break;
                }
                log("Word: " + word);

                JSONArray synsetNames = words.optJSONArray(word);
                if (synsetNames == null)
                {
                    log("This word doesn't exist, checking synsets...");
                    synsetNames = new JSONArray();
                    synsetNames.put(word);
                }
                
                for (int i = 0; i < synsetNames.length(); i++)
                {
                    String synsetName = synsetNames.get(i).toString();
                    JSONObject synset = synsets.optJSONObject(synsetName);
                    if (synset == null) {
                        log("This synset doesn't exist!");
                        continue;
                    }

                    String name = synset.optString("name");
                    String definition = synset.optString("definition");
                    String pos = synset.optString("pos");
                    
                    JSONArray examples = synset.optJSONArray("examples");
                    JSONArray synonyms = synset.optJSONArray("synonyms");
                    JSONArray antonyms = synset.optJSONArray("antonyms");

                    JSONArray hypernyms = synset.optJSONArray("hypernyms");
                    JSONArray hyponyms = synset.optJSONArray("hyponyms");

                    JSONArray part_meronyms = synset.optJSONArray("part_meronyms");
                    JSONArray substance_meronyms = synset.optJSONArray("substance_meronyms");

                    JSONArray part_holonyms = synset.optJSONArray("part_holonyms");
                    JSONArray substance_holonyms = synset.optJSONArray("substance_holonyms");

                    JSONArray entailments = synset.optJSONArray("entailments");

                    log("\nSense " + (i + 1) + ": " + name);
                    log("    synset: " + synsetName);
                    log("    definition: " + definition);
                    log("    pos: " + pos);
                    log("    examples: " + examples);
                    
                    log("    synonyms: " + synonyms);
                    log("    antonyms: " + antonyms);
                    
                    log("    hypernyms: " + hypernyms);
                    log("    hyponyms: " + hyponyms);
                    
                    log("    part_meronyms: " + part_meronyms);
                    log("    substance_meronyms: " + substance_meronyms);
                    
                    log("    part_holonyms: " + part_holonyms);
                    log("    substance_holonyms: " + substance_holonyms);

                    log("    entailments: " + entailments);
                    
                }
            }
            log("Done.");
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Reads the file.
     * 
     * @param file The file to read.
     * @param encoding The file encoding.
     * 
     * @return The file content as a string.
     */
    public static String readFile(File file, String encoding) throws IOException
    {
        if (!file.exists())
        {
            throw new IOException("Cannot open input file: " + file.getAbsolutePath());
        }
        log("Reading file: " + file.getAbsolutePath());
        FileInputStream inStream = new FileInputStream(file);
        int inBytes = inStream.available();
        byte[] inBuf = new byte[inBytes];
        int bytes = inStream.read(inBuf, 0, inBytes);
        inStream.close();
        log("Finished reading " + bytes + " bytes from input file: " + file.getAbsolutePath());
        String doc = new String(inBuf, encoding);
        return doc;
    }

    static void log(String message)
    {
        System.out.println(message);
    }

}